## The CDL Header API Reference Guide

Starting from PnetCDF 1.14.1, a set of C APIs have been added that read and
parse the header section of a CDL file.
([Common Data Language](https://docs.unidata.ucar.edu/nug/2.0-draft/cdl.html))
is a text notation for representing the structure and data of a binary netCDF
file. As CDL files are in a plain text format, they can can be read by a human.
The new APIs allow users to query the metadata defined in the CDL file. They
can be useful for programs that want to read only the file header of a netCDF
file produced by other applications. A CDL file can be generated by running the
PnetCDF utility program
[ncmpidump](https://github.com/Parallel-NetCDF/PnetCDF/tree/master/src/utils#ncmpidump)
or NetCDF's
[ncdump](https://docs.unidata.ucar.edu/nug/current/netcdf_utilities_guide.html#ncdump_guide).
Both utility programs take an command-line option "-h" to dump only the file
header in a plain text format.

Example programs can be found in
[examples/create_from_cdl.c](../examples/C/create_from_cdl.c) and
[benchmarks/WRF-IO/wrf_io.c.c](../benchmarks/WRF-IO/wrf_io.c).

---
### Table of contents

* [cdl_hdr_open](#cdl_hdr_open) opens and parses the CDL file's header
* [cdl_hdr_inq_format](#cdl_hdr_inq_format) returns the file format version
* [cdl_hdr_inq_ndims](#cdl_hdr_inq_ndims) returns the number of dimensions
  defined in CDL file
* [cdl_hdr_inq_dim](#cdl_hdr_inq_dim) returns the metadata of a dimension
* [cdl_hdr_inq_nvars](#cdl_hdr_inq_nvars) returns the number of variables
* [cdl_hdr_inq_var](#cdl_hdr_inq_var) returns the metadata of a variable
  defined in CDL file
* [cdl_hdr_inq_nattrs](#cdl_hdr_inq_nattrs) returns the number of attributes of
  a given variable
* [cdl_hdr_inq_attr](#cdl_hdr_inq_attr) returns the metadata of an attribute
* [cdl_hdr_close](#cdl_hdr_close) closes the CDL file

---
### cdl_hdr_open()
This API opens a file that is in the CDL format.

#### Usage
  ```
  int cdl_hdr_open(const char *filename,
                   int        *hidp);
  ```
  + **path** - [input] File name for a CDL file to be opened.

  + **hidp** - [output] Pointer to location where returned file ID is to be
    stored.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_ENFILE` - too many files opened.
  + `NC_ENOENT` - the specified input file does not exist.
  + `NC_EFILE` - other unknown I/O error, including unable to open the file.
  + `NC_ENOMEM` - unable to allocate memory (out of memory).
  + `NC_ENOTNC` - input file is not in the CDL format.

#### Example
  ```
  #include <pnetcdf.h>

  int err, hid;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);
  if (err != NC_NOERR)
      printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  ```

---
### cdl_hdr_inq_format()
This API returns the file format version of a given file ID.

#### Usage
  ```
  int cdl_hdr_inq_format(int  hid,
                         int *formatp);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.
  + **formatp** - [output] Pointer to location for returned format version, one
    of the following constants define in the header file pnetcdf.h. When this
    argument is NULL, this API takes no effect simply returns `NC_NOERR`.

    * 1 -  the classic CDF-1 format.
    * 2 -  the classic CDF-2 format (NC_64BIT_OFFSET).
    * 5 -  the classic CDF-5 format (NC_64BIT_DATA).

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file.

#### Example
  ```
  #include <pnetcdf.h>

  int err, hid, format;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* retrieve file format information of the input file */
  err = cdl_hdr_inq_format(hid, &format);
  if (err != NC_NOERR)
      printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  ```

---
### cdl_hdr_inq_ndims()
This API returns number of dimensions defined in the CDL file.

#### Usage
  ```
  int cdl_hdr_inq_ndims(int *hid,
                        int *ndimsp);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.
  + **ndimsp** - [output] Pointer to location for returned number of dimensions
    defined in the CDL file. When this argument is NULL, this API takes no
    effect simply returns `NC_NOERR`.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file.

#### Example
  ```
  #include <pnetcdf.h>

  int err, hid, ndims;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* retrieve the number of dimensions defined in the CDL file */
  err = cdl_hdr_inq_ndims(hid, &ndims);
  if (err != NC_NOERR)
      printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  ```

---
### cdl_hdr_inq_dim()
This API returns metadata of a dimension.

#### Usage
  ```
  int cdl_hdr_inq_dim(int          hid,
                      int          dimid,
                      char       **namep,
                      MPI_Offset  *sizep);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.
  + **dimid** - [input] dimension ID.
  + **namep** - [output] returned the name of dimension specified by its ID,
    'dimid'. Note users should not alter the contents of character string
    pointed by 'namep', as it is an internal allocated data structure. When
    `NULL` is passed in, this argument is ignored.
  + **sizep** - [output] size of the dimension. When `NULL` is passed in, this
    argument is ignored.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file, or or the value of `dimid` is not within the range of number of
    dimension defined in the CDL file.

#### Example
  ```
  #include <pnetcdf.h>

  int i, err, hid, ndims,

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* retrieve the number of dimensions defined in the CDL file */
  err = cdl_hdr_inq_ndims(hid, &ndims);

  for (i=0; i<ndims; i++) {
      char *name;
      MPI_Offset dimlen;

      /* retrieve metadata of dimension i */
      err = cdl_hdr_inq_dim(hid, i, &name, &dimlen);
      if (err != NC_NOERR)
          printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  }
  ```

---
### cdl_hdr_inq_nvars()
This API returns the number of variables defined in the CDL file.

#### Usage
  ```
  int cdl_hdr_inq_nvars(int  hid,
                        int *nvarsp);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.
  + **nvarsp** - [output] Pointer to location for returned number of variables
    defined in the CDFL file. When this argument is NULL, this API takes no
    effect simply returns `NC_NOERR`.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file.

#### Example
  ```
  #include <pnetcdf.h>

  int err, hid, nvars;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* retrieve number of variables defined in the CDL file */
  err = cdl_hdr_inq_nvars(hid, &nvars);
  if (err != NC_NOERR)
      printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  ```

---
### cdl_hdr_inq_var()
This API returns metadata of a variable defined in CDL file.

#### Usage
  ```
  int cdl_hdr_inq_var(int       hid,
                      int       varid,
                      char    **namep,
                      nc_type  *xtypep,
                      int      *ndimsp,
                      int     **dimidsp);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.
  + **varid** - [input] variable ID.
  + **namep** - [output] returned the name of variable specified by its ID,
    'varid'. Note users should not alter the contents of character string
    pointed by 'namep', as it is an internal allocated data structure. When
    `NULL` is passed in, this argument is ignored.
  + **xtypep** - [output] pointer to location for returned variable's data
    type, one of the set of predefined netCDF external data types. The type of
    this parameter, `nc_type`, is defined in the PnetCDF header file. The valid
    data types are `NC_BYTE`, `NC_CHAR`, `NC_SHORT`, `NC_INT`, `NC_FLOAT`,
    `NC_DOUBLE`, `NC_UBYTE`, `NC _USHORT`, `NC_UINT`, `NC_INT64`, and
    `NC_UINT64`. When `NULL` is passed in, this argument is ignored.
  + **ndimsp** - [output] pointer to location for returned number of dimensions
    the variable was defined as using. When `NULL` is passed in, this argument
    is ignored.
  + **dimidsp** - [output] returned vector of dimension IDs corresponding to
    the variable dimensions. Note users should not alter the contents pointed
    by argument `dimidsp`, as it is an internal allocated data structure. When
    `NULL` is passed in, this argument is ignored.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file, or the value of `varid` is not within the range of number of
    variables defined in the CDL file.

#### Example
  ```
  #include <pnetcdf.h>

  int i, err, hid, nvars;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* retrieve number of variables defined in the CDL file */
  err = cdl_hdr_inq_nvars(hid, &nvars);

  for (i=0; i<nvars; i++) {
      char *name;
      int ndims, *dimids;
      nc_type xtype;

      /* retrieve metadata of variable i defined in the CDL file */
      err = cdl_hdr_inq_var(hid, i, &name, &xtype, &ndims, &dimids);
      if (err != NC_NOERR)
          printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  }
  ```

---
### cdl_hdr_inq_nattrs()
This API returns number of attributes associated with a given variable.

#### Usage
  ```
  int cdl_hdr_inq_nattrs(int  hid,
                         int  varid,
                         int *nattrsp);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.
  + **varid** - [input] variable ID.
  + **nattrsp** - [output] pointer to location for returned number of
    attributes associated to the variable was defined in the CDL file. When
    `NULL` is passed in, this argument is ignored.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file, or the value of `varid` is not within the range of number of
    variables defined in the CDL file.

#### Example
  ```
  #include <pnetcdf.h>

  int err, hid;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* retrieve the number of global attributes */
  err = cdl_hdr_inq_nattrs(hid, NC_GLOBAL, &nattrs);
  if (err != NC_NOERR)
      printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));

  for (i=0; i<nvars; i++) {
      int nattrs;

      /* retrieve the number of attributes associated with variable i */
      err = cdl_hdr_inq_nattrs(hid, i, &nattrs);
      printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  }
  ```

---
### cdl_hdr_inq_attr()
This API returns metadata of an attribute.

#### Usage
  ```
  int cdl_hdr_inq_attr(int          hid,
                       int          varid,
                       int          attrid,
                       char       **namep,
                       nc_type     *xtypep,
                       MPI_Offset  *nelemsp,
                       void       **valuep);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.
  + **varid** - [input] variable ID.
  + **attrid** - [input] attribute ID.
  + **namep** - [output] returned the name of attribute specified by its ID,
    'attrid'. Note users should not alter the contents of character string
    pointed by 'namep', as it is an internal allocated data structure. When
    `NULL` is passed in, this argument is ignored.
  + **xtypep** - [output] pointer to location for returned attribute's data
    type, one of the set of predefined netCDF external data types. The type of
    this parameter, `nc_type`, is defined in the PnetCDF header file. The valid
    data types are `NC_BYTE`, `NC_CHAR`, `NC_SHORT`, `NC_INT`, `NC_FLOAT`,
    `NC_DOUBLE`, `NC_UBYTE`, `NC _USHORT`, `NC_UINT`, `NC_INT64`, and
    `NC_UINT64`. When `NULL` is passed in, this argument is ignored.
  + **nelemsp** - [output] pointer to location for returned number of elements
    the attribute was defined as using. When `NULL` is passed in, this argument
    is ignored.
  + **valuep** - [output] pointer to location for returned attribute value(s).
    Note users should not alter the contents pointed by argument `valuep`, as
    it is an internal allocated data structure. When `NULL` is passed in, this
    argument is ignored.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file, the value of `varid` is not within the range of number of variables
    defined in the CDL file, or the value of `attrid` is not within the range
    of number of attributes associated to the variable.

#### Example
  ```
  #include <pnetcdf.h>

  int err, hid, nattrs, nvars, nelems;
  char *name;
  void *value;
  nc_type xtype;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* retrieve the number of global attributes */
  err = cdl_hdr_inq_nattrs(hid, NC_GLOBAL, &nattrs); ERR

  for (i=0; i<nattrs; i++) {
      /* retrieve metadata of global attribute i */
      err = cdl_hdr_inq_attr(hid, NC_GLOBAL, i, &name, &xtype, &nelems, &value);
      ERR
  }

  /* retrieve number of variables defined in the CDL file */
  err = cdl_hdr_inq_nvars(hid, &nvars);

  for (i=0; i<nvars; i++) {
      /* retrieve the number of attributes associated with variable i */
      err = cdl_hdr_inq_nattrs(hid, i, &nattrs); ERR

      for (j=0; j<nattrs; j++) {
          /* retrieve metadata of attribute j associated with variable i */
          err = cdl_hdr_inq_attr(hid, i, j, &name, &xtype, &nelems, &value);
          if (err != NC_NOERR)
              printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
      }
  }
  ```

---
### cdl_hdr_close()
This API closes the CDL file.

#### Usage
  ```
  int cdl_hdr_close(int hid);
  ```
  + **hid** - [input] file ID, from a previous call to `cdl_hdr_open`.

#### Return Error Codes
This API returns the value `NC_NOERR` if no errors occurred. Otherwise, the
returned value of type `int` indicates an error. Possible error codes and
causes of errors include:
  + `NC_EBADID` - the specified file ID, `hid`, does not refer to an opened CDL
    file.

#### Example
  ```
  #include <pnetcdf.h>

  int err, hid;

  /* open the input file in CDL format */
  err = cdl_hdr_open("cdl_header.txt", &hid);

  /* close CDL file */
  err = cdl_hdr_close(&hid);
  if (err != NC_NOERR)
      printf("Error at %s:%d : %s\n", __FILE__,__LINE__, ncmpi_strerror(err));
  ```

